classdef vct3Array
    % classdef vct3 - Arrays of cartesian 3 vectors, intended to
    % be used with classes vct3, RotMX and Frame.
    %
    % Copyright (C) Russell H. Taylor, Johns Hopkins University, 2013
    %
    % Properties
    %   el(3,N) is a 3xN matrix of column vectors, each corresponding to a
    %           cartesian vector
    %
    % Notation 
    %   N                           number of column vectors
    %   v, v1, v2                   vct3 objects
    %   V, V1, V2, ...              vct3Array objects
    %   s, x,y,z,a,b                scalar objects
    %   M, M1, M2, ...              Matrix object 
    %   X                           May be matrix or scalar (see V1.*V2)
    %   k                           Integer
    %   S                           Subscript object
    %
    % Constructors
    %   V = vct3Array()             Null array 
    %   V = vct3Array(N)            matrix of N vct3 objects
    %   V = vct3Array(V)                 copy constructor
    %   V = vct3(M)                 creates vct3 from 3xN matrix M
    %   V = unit(V1)                returns V/norm(V)
    %
    %   V = vct3Array.null()        []
    %   V = vct3Array.rand(N,a,b)   creates matrix of N random vct3 objects
    %                               whose elements are between a and b
    %   V = vct3Array.rand(N)       equivalent to vct3.rand(N,0,1)
    %
    % Indexing and access methods
    %   k = NumEl(V)                returns N
    %   v = V(k)                    returns vct3(V1.el(:,k)) 
    %   V(k) = X                    sets k'th columns to X
    %   M = X(V1)                   computes V1.el(1,:)
    %   M = Y(V1)                   computes V1.el(2,:)
    %   M = Z(V1)                   computes V1.el(3,:)
    %
    %   V = [V1, V2] V=[V1,V2]      returns vct3Array(V1.el,V2.el)  
    %   V = horzcat(V1,V2)          returns vct3Array(V1.el,V2.el)
    %
    % Computational methods (here if V2 is vct3, treats as vct3Array(1)
    %
    %   V = V1+V2                   computes V1+V2
    %   V = V1-V2                   computes V1-V2
    %   V = -V1                     computes -V1
    %   V = V1*V2                   computes V1*V2 = V1.el'*V2.el
    %   V = V1*s                    computes V1*s  = vct3(V1*s)
    %   X = V1*M                    computes V1*M  = V1.el'*M
    %   V = V1.*V2                  computes V1.*V2= vct3(V1.el*V2.el)
    %   V = V1.*X                   computes V1.*V2= vct3(V1.el*X)
    %   V = V1./s                   computes V1./s = vct3(V1.el./s)
    %   
    %   V = plus(V1,V2)             computes V1+V2
    %   V = minus(V1,V2)            computes V1-V2
    %   V = uminus(V1)              computes -V1
    %   M = mtimes(V1,V2)           computes V1*V2 = V1.el'*V2.el
    %   V = mtimes(V1,s)            computes V1*s  = vct3(V1*s)
    %   X = mtimes(V1,M)            computes V1*M  = V1.el'*M
    %   V = times(V1,V2)            computes V1.*V2= vct3(V1.el*V2.el)
    %   V = times(V1,X)             computes V1.*s = vct3(V1.el.*X)
    %   V = rdivide(V1,s)           computes V1./s = vct3(V1.el./s)
    %
    %   V = abs(V)                  computes vct3(abs(V1.el))
    %   v = mean(V1)                computes vct3(mean(V1.el,2))
    %   M = cov(V1)                 computes 3x3 cov(V1)
    %   M = norm(V1,p)              computes norm(V1.el,p)
    %   V = unit(V1)                returns vct3Array(V1/norm(V1)
    %    
    %
    % Other methods
    %   display(V)                  console display (calls disp)
    %   disp(V)                     basic console display
    % 
    % Copyright (C) Russell H. Taylor 2013
    % For use with CIS I only
    % Do not redistribute without written permission from Russell Taylor
              
    
    properties
        el = []'
    end
    
    methods
        function V = vct3Array(varargin)
            switch nargin
                case 0
                    return; % default is null vct3
                case 1
                    if isa(varargin{1},'vct3')
                        sz = size(varargin{1});
                        if sz==[1,1]
                            V.el = varargin{1}.el;
                        else
                            if sz(1)==1
                                V.el=zeros(3,sz(2));
                                for j=1:sz(2)
                                    V.el(:,j)=varargin{1}(j).el
                                end
                            else
                                error('BadArg','Cannot handle this vct3Array case',varargin)
                            end
                        end
                    elseif isa(varargin{1},'vct3Array')
                        V.el = varargin{1}.el;
                    elseif isscalar(varargin{1})
                        V.el = zeros(3,varargin{1});
                    else 
                        sz = size(varargin{1});
                        if sz(1) == 3
                            V.el = varargin{1};
                        elseif sz(2)==3
                            V.el = varargin{1}';
                        else
                            error('BadArg','Bad argument to vct3Array',varargin);
                        end
                    end
                    return;
                otherwise
                    error('BadArg','Wrong number of arguments to vct3Array',nargin);
            end
        end
        
        function k = NumEl(V)
            k = size(V.el,2);
        end

        function V = subsref(V1,S)
            switch S(1).type 
                case '()'
                    SS = substruct('.','el','()',{':',S.subs{:}});
                    V = vct3(builtin('subsref',V1,SS));
                    return;
                otherwise
                    V=builtin('subsref',V1,S);
            end
        end
            
        function V = subsasgn(V,S,X)
           switch S(1).type
               case '()'
                   if isa(X,'vct3') || isa(X,'vct3Array')
                       SS = substruct('.','el','()',{':',S.subs{:}});
                       V = builtin('subsasgn',V,SS,X.el);
                   elseif size(V.el,2)==1 && isscalar(X)
                       SS = substruct('.','el','()',{S.subs{:},[1]});
                       V = builtin('subsasgn',V,SS,X);                       
                   else
                       error('BadArg','Bad right hand side value for subsasgn',X);
                   end
               otherwise
                   V = builtin('subsasgn',V,S,X);
           end
        end    
        
        function s = X(V)
            s = V.el(1,:)
        end
       
        function s = Y(V)
            s = V.el(2,:)
        end
        
               
        function s = Z(V)
            s = V.el(3,:)
        end
        
        function V = horzcat(V1,V2)
            V = vct3Array([V1.el,V2.el]);
        end
        
        function s = mtimes(V1,V2)
            if isa(V2,'vct3Array')
                s = V1.el'*V2.el; 
            elseif isa(V2,'vct3')
                s = V1.el'*V2.el; 
            elseif isscalar(V2)
                s = vct3(V1.el*V2);
            else
               s = V1.el'*V2.el;
            end;                     
        end
        
        function v3 = times(V1,V2)
             if isa(V2,'vct3Array')
                v3 =vct3Array(V1.el.*V2.el); 
             elseif isa(V2,'vct3')
                v3 =vct3Array(V1.el.*V2.el);
             else
                v3 =vct3Array(V1.el.*V2);
             end;
        end
        
        function v3 = uminus(V1)
            v3 = vct3Array(-V1.el);
        end
        
        function V = abs(V1)
            V = vct3Array(abs(V1.el));
        end
        
        function v3 = plus(V1,V2)
            v3 = vct3Array(bsxfun(@plus,V1.el,V2.el)); % add type checking later
        end
        
        function v3 = minus(V1,V2)
            v3 = vct3Array(bsxfun(@minus,V1.el,V2.el)); % add type checking later
        end
        
        function V = mrdivide(V1,s)
            V = vct3Array(V1.el/s);
        end
        
        function V = rdivide(V1,s)
            V = vct3Array(V1.el/s);
        end
        
        function result = mean(V)
            result = vct3(mean(V.el,2));
        end
        
        function result = cov(V)
            result = cov(V.el')';
        end
        
        function F = covFrame(V)
            p = V.mean();
            C = V.cov();
            [Q,D] = eigs(C,3);
            R = [Q(:,1),cross(Q(:,3),Q(:,1)),Q(:,3)];
            F = Frame(RotMx(R),p);          
        end
 
        function V = unit(V1)
            A = arrayfun(@(i)V1.el(:,i)/norm(V1.el(:,i)),1:size(V1.el,2), ...
                                'UniformOutput',0);
            V = vct3Array(cell2mat(A));
        end
        
        function s = norm(varargin)
            sz = size(varargin{1}.el);
            switch nargin
                case 1
                    p = 2;
  
                case 2
                    p = varargin{2};
                otherwise
                    error('BadArg','Wrong number of arguments to vct3Array.norm',nargin);
            end
            s = arrayfun(@(i)norm(varargin{1}.el(:,i),p),1:sz(2));
        end
         
        
        function disp(V)
            disp(V.el);
        end
        
        function display(V)          
            if isequal(get(0,'FormatSpacing'),'compact')
               disp([inputname(1) ' =']);
               disp(V);
            else
               disp(' ')
               disp([inputname(1) ' =']);
               disp(' ');
               display(V.el);
            end
        end
               
    end
    
    methods (Static)
        function V = null()
            V = vct3Array([]);
        end
        
        function V = rand(N,a,b)
            switch nargin
                case 0
                    V = vct3Array(0);
                    return;
                case 1
                    V = vct3Array.rand(N,0,1);
                case 2
                    if isa(a,'vct3')
                        ae = a.el; abe = 2*ae;
                        Ve = [-ae(1)+abe(1).*rand(1,N); ...
                              -ae(2)+abe(2).*rand(1,N); ...
                              -ae(2)+abe(2).*rand(1,N)];
                        V = vct3Array(Ve);
                    else
                        V = vct3Array(-a +2*a.*rand(3,N));
                    end
                case 3
                     if isa(a,'vct3')
                        ae = a.el; abe = b.el-a.el;
                        Ve = [ae(1)+abe(1).*rand(1,N); ...
                              ae(2)+abe(2).*rand(1,N); ...
                              ae(2)+abe(2).*rand(1,N)];
                        V = vct3Array(Ve);
                     else
                        V = vct3Array(a +(b-a).*rand(3,N));
                    end
                otherwise
                    error('BadArg','Wrong number of arguments to vct3.rand',nargin);
            end
       end
   end
    
end

